/** A stack is a list that follows the last-in-first-out (LIFO)
 * principle. An example is a stack of cafeteria trays. You take
 * the top one, which is the last one that the staff put in.<br><br>
 
 * Class LinkedListStack provides a linked-list implementation of
 * a stack. Each cell or node of the linked list is an object of
 * class StackCell, which appears at the end of this file. It is not
 * public a source file can have nly one public class. <br><br>
 * 
 * In addition, to show how to do such things, each implementation
 * provides operation not usually found with stacks:<br><br>
 * 
 * 1. Delete the top occurrence of an element e (if present)<br>
 * 
 * 2. Tell whether the stack contains a particular element<br>
 * 
 * 3. Reverse the stack --turn it upside down, r backwards, depending
 *    on how you look at it. We provide iterative and recursive methods
 *    for reversing the stack. */
public class LinkedListStack<T> {

    // Head (first cell) of the List --null if list empty.
    // When viewed as stack, the first element on the list is the top
    // and the last element on the list is the bottom.
    private StackCell<T> head; 

    /**Constructor: An empty stack. */
    public LinkedListStack() {
        head= null;
    }

    /** Push e on this stack. */
    public  void push(T e) {
        head= new StackCell<T>(e, head);
    }

    /** Pop the top element of this stack and return it.
     * Throw RuntimeException if the stack is empty */
    public  T pop(T e) {
        if (head == null)
            throw new RuntimeException("Cannot pop an empty stack");
        T h= head.datum;
        head= head.next;
        return h;
    }

    /** Return the number of elements. */
    public  int size() {
        int count= 0;
        StackCell<T> current= head;
        while (current != null) {
            count= count + 1;
            current= current.next;
        }
        return count;
    }

    /** Delete first (topmost) occurrence of e from this stack (if it is there). */
    public void delete(T e) {
        if (head == null) return;
        if (equal(e, head.datum)) {
            head= head.next;
            return;
        }

        StackCell<T> current= head;
        StackCell<T> scout= head.next;
        // invariant: the cells head, head.next, ..., current
        //            do not contain e
        //            and scout = current.next.
        while (scout != null) {
            if (equal(e, scout.datum)) {
                current.next= scout.next; // found--unlink cell
                return;
            }
            current= scout;
            scout= current.next;
        }
    }

    /** Return true iff x and y are both null or
     * x is not null and x.equals(y). */
    private boolean equal(T x, T y) {
        return (x == null && y == null) || (x != null && x.equals(y));
    }

    /** Return true iff this stack contains e.  */
    public boolean contains(T e) {
        StackCell<T> current= head;
        // inv: e is not in head, head.next, ..., cell before current
        while (current != null  &&  !equal(e, current.datum)) {
            current= current.next;
        }
        return current != null;
    }

    /** Return the representation of this stack in this form:<br>
     * [ e0, e1, ..., elast]<br>
     * where e0 is the top element and elast is the bottom one. */
    public String toString() {
        String result = "[";
        StackCell<T> current = head;
        while (current != null) {
            if (current != head)
                result= result + ", ";
            result= result + current.datum;
            current = current.next;
        }
        return result + "]";
    }

    /** Reverse this list in place. */
    public void ReverseInPlace() {
        // Initial list is head -> A -> B -> ... -> C -> D -> ... -> E
        // Final   list is         A <- B <- ... <- C <- D <- ... <- E <-head

        StackCell<T> rest= head;
        head= null;
        // invariant:              A <- B <- ... <- C <- head
        //                 rest ->                       D -> ... -> E
        
        // So each iteration removes first node of rest and pushes it on head
        while (rest != null) {
            StackCell<T> temp= rest;
            rest= rest.next;
            temp.next= head;
            head= temp;
        }
    }
    
    /** Reverse this list in place. */
    public void ReverseInPlaceRecursive() {
        StackCell<T> c= head;
        head= null;
        prependInPlace(c);
    }

    /** Prepend the elements of list c to this list */
    private void prependInPlace(StackCell<T> c) {
        if (c == null) return;
        StackCell<T> t= c;
        c= c.next;
        t.next= head;
        head= t;
        prependInPlace(c);
    } 
}

/** Stack cell (a helper class; visible to other classes in same package). */
class StackCell<T> {

   public T datum; // Data for this cell
   public StackCell<T> next; // Next cell in this Stack

   /** Constructor: an instance with datum d and next cell next. */
   public StackCell(T d, StackCell<T> next) {
      datum= d;
      this.next= next;
   }
}
